// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "dsp_core.h"
#include "hw_ai.h"

static const char* reg_names[] = 
{
	// c0
	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,
	NULL,	"DSCR",	NULL,	"DSBL",	NULL,	"DSPA",	"DSMAH","DSMAL",
	// d0
	NULL,	NULL,	NULL,	NULL,	"ACSAH","ACSAL","ACEAH","ACEAL",
	"ACCAH","ACCAL",NULL,	NULL,	NULL,	NULL,	NULL,	NULL,
	// e0
	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,
	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	"AMDM",
	// f0
	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,
	NULL,	NULL,	NULL,	"DIRQ",	"DMBH", "DMBL", "CMBH", "CMBL",
};

union DSP_DWORD {
	DWORD d;
	struct { WORD lo, hi; };
};

struct DMA {
	DSP_DWORD mainAddr;
	WORD len, dspAddr, control;
};
static DMA sDma;

void DSPInterpreter::hwWrite(WORD address, WORD data) {
	if(address < DSP_HWMEMBASE)
		throw hardware_fatal_exception("DSP hwWrite out of bounds");
	BYTE offset = (BYTE)address;
	DSPLOG("hwWrite %02X(%s), %04X\n", offset, reg_names[offset-0xC0], data);
	switch(offset) {
	case 0xFB:
		if(data & 1) {
			HwAI::assert_dsp_interrupt(&h);
		}
		break;
	case 0xFC:
		h.hwh(PR_CMBH, data & 0x7FFF);
		break;
	case 0xFD:
		h.hwh(PR_CMBL, data);
		h.hwh(PR_CMBH, h.hrh(PR_CMBH) | 0x8000);
		DSPLOG("Mail to CPU detected: 0x%08X\n", h.hrw(PR_CMBH));
		//HwAI::assert_dsp_interrupt(&h);
		break;
	case 0xCE: sDma.mainAddr.hi = data; break;
	case 0xCF: sDma.mainAddr.lo = data; break;
	case 0xCD: sDma.dspAddr = data; break;
	case 0xC9: sDma.control = data; break;
	case 0xCB:
		sDma.len = data;
		DSPLOG("DSP mainmem DMA ");
		if(sDma.control & 1) {
			DSPLOG("from ");
		} else {  //to dsp
			DSPLOG("to ");
		}
		if(sDma.control & 2) {
			DSPLOG("iram");
		} else {
			DSPLOG("dram");
		}
		DSPLOG(". mainAddr: 0x%08X dspAddr: 0x%04X len: 0x%X(%i)\n",
			sDma.mainAddr.d, sDma.dspAddr, sDma.len, sDma.len);
		if(sDma.control & 1) {  //to main
			if(sDma.control & 2) {  //iram
				//iDmaRead(sDma.dspAddr,
					//mainmem.getp_translated(sDma.mainAddr.d, sDma.len), sDma.len);
				throw hardware_fatal_exception("DSP DMA from IRAM");
			} else {	//dram
				dDmaRead(sDma.dspAddr,
					mainmem.getp_translated(sDma.mainAddr.d, sDma.len), sDma.len);
			}
		} else {  //to dsp
			if(sDma.control & 2) {  //iram
				iDmaWrite(sDma.dspAddr,
					mainmem.getp_translated(sDma.mainAddr.d, sDma.len), sDma.len);
			} else {	//dram
				dDmaWrite(sDma.dspAddr,
					mainmem.getp_translated(sDma.mainAddr.d, sDma.len), sDma.len);
			}
		}
		break;
	default:
		throw hardware_fatal_exception("DSP hwWrite unemulated");
	}
}
WORD DSPInterpreter::hwRead(WORD address) {
	if(address < DSP_HWMEMBASE)
		throw hardware_fatal_exception("DSP hwRead out of bounds");
	BYTE offset = (BYTE)address;
	DSPLOG("hwRead %02X(%s)", offset, reg_names[offset-0xC0]);
#define HWRETURN(d) { WORD temp = (d); DSPLOG(", %04X\n", temp); return temp; }
	switch(offset) {
	case 0xFC:
		{
			WORD data = h.hrh(PR_CMBH);
			if(data & 0x8000)	//mail is waiting to be delivered to CPU
				m.pause = true;
			HWRETURN(data);
		}
	case 0xFD:
		HWRETURN(h.hrh(PR_CMBL));
	case 0xFE:
		{
			WORD data = h.hrh(PR_DMBH);
			if(!(data & 0x8000))	//no mail for us yet
				m.pause = true;
			HWRETURN(data);
		}
	case 0xFF:
		h.hwh(PR_DMBH, h.hrh(PR_DMBH) & 0x7FFF);
		HWRETURN(h.hrh(PR_DMBL));
	case 0xC9:
		HWRETURN(sDma.control);
	default:
		DSPLOG("\n");
		throw hardware_fatal_exception("DSP hwRead unemulated");
	}
}
